#!/usr/bin/python

#
# Small program to examine positional data from a VRPN tracker
#
# Requirements:
#    apt-get install python-qt python-qt-gl
#

import sys
import math
import time
import argparse

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from PyQt4.QtOpenGL import *

from OpenGL.GL import *
from OpenGL.GLU import *

sys.path.append(".")

import vrpn_Tracker

VERBOSE = False

#
class GLCamera:
    def __init__(self):
        self.angle = 0
        self.radius = 8
        self.height = 2
        self.target = QVector3D(0, 0, 0)
        pass

    def gluLookAt(self):
        p = self.pos()
        t = self.target
        gluLookAt(p.x(), p.y(), p.z(),
                  t.x(), t.y(), t.z(),
                  0.0, 1.0, 0.0)
        pass

    def pos(self):
        x = self.target.x() + self.radius * math.sin(self.angle)
        y = self.target.y() + self.height
        z = self.target.z() + self.radius * math.cos(self.angle)
        return QVector3D(x, y, z)

    pass

#
def convert_to_glmatrix(pos, quat):
    m = [1,0,0,0,
         0,1,0,0,
         0,0,1,0,
         0,0,0,1]

    p4_2 = quat[0]*2;  p5_2 = quat[1]*2;  p6_2 = quat[2]*2;
    xx = p4_2*quat[0]; xy = p4_2*quat[1]; xz = p4_2*quat[2];
    yy = p5_2*quat[1]; yz = p5_2*quat[2];
    zz = p6_2*quat[2];

    sx = quat[3]*p4_2; sy = quat[3]*p5_2; sz = quat[3]*p6_2;

    m[0] =1.0-(yy+zz);  m[4] =    (xy-sz);  m[8] =    (xz+sy);
    m[1] =    (xy+sz);  m[5] =1.0-(xx+zz);  m[9] =    (yz-sx);
    m[2] =    (xz-sy);  m[6] =    (yz+sx);  m[10]=1.0-(xx+yy);
    m[3] =0          ;  m[7] =0          ;  m[11]=0          ;

    m[12] = pos[0]
    m[13] = pos[1]
    m[14] = pos[2]
    m[15] = 1.0
    return m

#
def draw_grid(glw, n):
    glColor4f(0.2, 0.2, 0.2, 0.2)
    glLineWidth(1)
    glBegin(GL_LINES)
    for i in range(-n,n):
        for j in range(0, 10):
            glVertex3f(i+j/10.0, 0,  n)
            glVertex3f(i+j/10.0, 0, -n)
            pass
        pass
    for i in range(-n,n):
        for j in range(0, 10):
            glVertex3f( n, 0, i+j/10.0)
            glVertex3f(-n, 0, i+j/10.0)
            pass
        pass
    glEnd()

    glColor4f(0.9, 0.9, 0.9, 0.9)
    glLineWidth(1)
    glBegin(GL_LINES)
    for i in range(-n,n+1):
        glVertex3f(i, 0,  n)
        glVertex3f(i, 0, -n)
        pass
    for i in range(-n,n+1):
        glVertex3f( n, 0, i)
        glVertex3f(-n, 0, i)
        pass
    glEnd()

    glColor4f(1, 1, 1, 1)
    for i in range(n):
        glw.renderText(0, 0, i, "%dm" % i)
        pass
    glw.renderText(n, 0, 0, "x")
    glw.renderText(0, 0, n, "z")
    pass

#
def draw_axes(scale):
    s = scale
    glLineWidth(2)
    glBegin(GL_LINES)
    glColor3f(1,0,0) # x - r
    glVertex3f(0,0,0)
    glVertex3f(s,0,0)
    glColor3f(0,1,0) # y - g
    glVertex3f(0,0,0)
    glVertex3f(0,s,0)
    glColor3f(0.2,0.2,1) # z - b
    glVertex3f(0,0,0)
    glVertex3f(0,0,s)
    glEnd()
    pass

def ray_plane_intersect(A, K, P, N):
    # plane eq. N * X = N * A = d
    # line eq.  X = A + Kt

    # solve N * (A + Kt) = d for t
    # t = (d - N * A) / (N * K)
    d = QVector3D.dotProduct(P, N)
    t1 = QVector3D.dotProduct(N, A)
    t2 = QVector3D.dotProduct(N, K);
    if t2: return A + K * ((d - t1) / t2)
    return None

# vector clamp
def clamp(v, vmin, vmax):
    x = max(vmin.x(), min(v.x(), vmax.x()))
    y = max(vmin.y(), min(v.y(), vmax.y()))
    z = max(vmin.z(), min(v.z(), vmax.z()))
    return QVector3D(x, y, z)

#
class ViewWidget3D(QGLWidget):
    def __init__(self, parent):
        QGLWidget.__init__(self, parent)
        self.setFocusPolicy(Qt.StrongFocus)

        self.camera = GLCamera()
        self.quadric = gluNewQuadric()
        self.mutex = QMutex()
        self.points = {}
        self.proj_matrix = None
        self.model_matrix = None
        self.viewport = None

        self.lastmousepos = None
        self.dragging = False
        self.drag_origin = None

        self.grid_size = 5
        pass

    def initializeGL(self):
        QGLWidget.initializeGL(self)
        glClearColor(0,0,0,1)
        pass

    def resizeGL(self, w, h):
        glViewport(0, 0, w, h)
        pass

    def paintGL(self):
        QGLWidget.paintGL(self)

        # erase previous frame
        glClear(GL_COLOR_BUFFER_BIT)

        # setup matrices
        glMatrixMode(GL_PROJECTION)
        glLoadIdentity()
        gluPerspective(45, self.width()/float(self.height()), 1, 100)

        glMatrixMode(GL_MODELVIEW)
        glLoadIdentity()
        self.camera.gluLookAt()

        self.proj_matrix = glGetDoublev(GL_PROJECTION_MATRIX)
        self.model_matrix = glGetDoublev(GL_MODELVIEW_MATRIX)
        self.viewport = glGetIntegerv(GL_VIEWPORT)

        # grid
        glPushMatrix()
        glColor4f(1,1,1,1)
        draw_grid(self, self.grid_size)
        glPopMatrix()

        # axes
        glViewport(0, 0, int(self.width() * 0.075), int(self.height() * 0.075))
        glPushMatrix()
        draw_axes(100)
        glPopMatrix()
        glViewport(0, 0, int(self.width()), int(self.height()))

        # points
        radius = 0.01
        self.mutex.lock()
        for id in self.points:
            p = self.points[id]
            try: pos, quat = p[0:3], p[3:]
            except: pass
            m = convert_to_glmatrix(pos, quat)
            glPushMatrix()
            glMultMatrixf(m)
            glColor4f(1,1,1,1)
            gluSphere(self.quadric, radius, 8, 8)
            self.renderText(radius, radius, radius, "%d"%(id,))
            # only draw axes if rotation is not identity
            if not (quat[0] == 0 and quat[1] == 0 and quat[2] == 0 and quat[3] == 1):
                draw_axes(0.1)
                pass
            glPopMatrix()
            pass
        self.mutex.unlock()
        pass

    def mouseMoveEvent(self, event):
        if self.lastmousepos == None:
            self.lastmousepos = QVector2D(event.x(), event.y())
            return
        newpos = QVector2D(event.x(), event.y())
        dt = newpos - self.lastmousepos
        self.lastmousepos = newpos

        if event.buttons() & Qt.RightButton:
            self.camera.angle -= dt.x() / self.width() * 2
            self.camera.height += dt.y() / self.height() * 2
            pass
        if event.buttons() & Qt.LeftButton:
            w = QVector2D(event.x(), self.viewport[3] - event.y()) #TODO test in mswindows
            v0 = QVector3D(*gluUnProject(w.x(), w.y(), 0, self.model_matrix, self.proj_matrix, self.viewport))
            v1 = QVector3D(*gluUnProject(w.x(), w.y(), 1, self.model_matrix, self.proj_matrix, self.viewport))
            v = v1 - v0
            v.normalize()
            v = ray_plane_intersect(self.camera.pos(), v, QVector3D(0, 0, 0), QVector3D(0, 1, 0))
            if self.drag_origin != None:
                delta = self.drag_origin - v
                self.camera.target = self.prev_target + delta
                self.drag_origin = None
                n = self.grid_size
                self.camera.target = clamp(self.camera.target, QVector3D(-n, 0, -n), QVector3D(n, 0, n))
                pass
            else:
                self.drag_origin = v
                self.prev_target = self.camera.target
                pass
            pass
        if event.buttons() & Qt.MiddleButton:

            pass

        self.update()
        pass

    def mousePressEvent(self, event):
        pass

    def mouseReleaseEvent(self, event):
        self.drag_origin = None
        self.lastmousepos = None
        self.update()
        pass

    def wheelEvent(self, event):
        if(event.delta() > 0):
            self.camera.radius *= 0.9
        else:
            self.camera.radius *= 1.1
            pass
        self.update()
        pass

    def keyReleaseEvent(self, event):
        if event.text() == 'q':
            pass
        self.update()
        pass

    def thandler(self, userdata, t):
        if VERBOSE:
            print userdata, t
            pass
        self.mutex.lock()
        try:
            self.points[t[0]] = t[1:]
            pass
        except:
            print "unknown data: ", t
            pass
        self.mutex.unlock()
        self.update()
        pass
    pass

#
class mainthread(QThread):
    def __init__(self, trackers, widget):
        QThread.__init__(self)
        self.trackers = trackers
        self.widget = widget
        pass
    def run(self):
        while True:
            for t in self.trackers:
                t.mainloop()
                pass
            time.sleep(0.01)
            pass
        pass
    pass

#
def main():
    global VERBOSE

    # parse command line
    parser = argparse.ArgumentParser()
    parser.add_argument("--verbose", dest="verbose", default=False, action="store_true")
    parser.add_argument("--device", default="Tracker0@localhost", action="store", help="Tracker, example: Tracker0@localhost")
    args = parser.parse_args()

    print "using tracker: ", args.device

    if args.verbose:
        print "verbose mode"
        VERBOSE = True
        pass
    # create QT widgets
    app = QApplication(sys.argv)
    mw = QMainWindow()
    widget = ViewWidget3D(mw)
    mw.setCentralWidget(widget)
    mw.resize(800,600)
    mw.show()

    # create VRPN tracker and register change handlers
    vrpn_Tracker.register_tracker_change_handler(widget.thandler)
    tracker = vrpn_Tracker.vrpn_Tracker_Remote(args.device)
    tracker.register_change_handler(None, vrpn_Tracker.get_tracker_change_handler())

    # start main thread
    mt = mainthread([tracker,], widget)
    mt.start()

    # start main app
    app.exec_()
    pass

main()
